<?php

namespace App\Services;

use Exception;
use Illuminate\Support\Facades\Http;
use InvalidArgumentException;
use Spatie\FlareClient\Http\Exceptions\BadResponse;

class JayaPayment
{
    const IDR_PAYMENTS = [
        'BCA', 'MANDIRI', 'PERMATA', 'CIMB', 'BNI', 'MAYBANK', 'DANAMON', 'BRI', 'BSI', 'BNC', 'OVO', 'DANA', 'LINKAJA', 'SHOPEEPAY', 'QRIS', 'GOPAY_QRIS', 'ALFAMART', 'TRANSFER_BCA'
    ];

    const API_URL = 'https://openapi.jayapayment.com';

    public static function inquiryIDR(array $credentials, array $payload)
    {
        try {
            self::validationCredentials($credentials);

            $payload['merchantCode'] = $credentials['merchant_id'];
            $payload['notifyUrl'] = $credentials['url_callback'];
            $payload['orderType'] = '0';

            $body = self::validationInquiryIDR($credentials['private_key'], $payload);

            $url = sprintf('%s/gateway/prepaidOrder', self::API_URL);
            $response = Http::acceptJson()->post($url, $body);
            if (!$response->ok() || $response->json('platRespCode') == 'FAIL') {
                throw new BadResponse($response->json('platRespMessage'), $response->status());
            }

            return [
                'status' => true,
                'code' => 200,
                'message' => 'OK',
                'data' => $response->json(),
            ];
        } catch (Exception $e) {
            return [
                'status' => false,
                'code' => $e->getCode(),
                'message' => $e->getMessage(),
                'data' => null,
            ];
        }
    }

    public static function callback(string $publicKey, array $payload): array
    {
        $dataSign = $payload['platSign'];
        unset($payload['platSign']);

        $decrypt = self::decrypt($dataSign, $publicKey);

        if (self::sortPayload($payload) != $decrypt || $payload['code'] != '00')

        return [
            'status' => true,
            'message' => 'SUCCESS',
        ];
    }
    
    public static function fee($amount, $isBank = true): array
    {
        $rateBank = (0.04 * $amount) + 6500;
    
        return [
            'amount' => $amount,
            'fee' => $fee = $isBank ? $rateBank : (0.05 * $amount),
            'net_amount' => $amount,
        ];
    }


    protected static function sortPayload(array $payload): string
    {
        ksort($payload);
        return implode('', $payload);
    }

    public static function sign($data, $privateKey)
    {
        $dataKey = sprintf("-----BEGIN PRIVATE KEY-----\n%s\n-----END PRIVATE KEY-----", $privateKey);
        $key = openssl_pkey_get_private($dataKey);

        $crypto = '';

        foreach (str_split($data, 117) as $chunk) {
            openssl_private_encrypt($chunk, $encryptData, $key);
            $crypto .= $encryptData;
        }

        return base64_encode($crypto);
    }

    public static function decrypt($data, $publicKey)
    {
        $data = base64_decode($data);
        $dataKey = sprintf("-----BEGIN PUBLIC KEY-----\n%s\n-----END PUBLIC KEY-----", $publicKey);
        $key =  openssl_pkey_get_public($dataKey);

        $crypto = '';

        foreach (str_split($data, 128) as $chunk) {
            openssl_public_decrypt($chunk, $decryptData, $key);
            $crypto .= $decryptData;
        }

        return $crypto;
    }

    protected static function types(string $method, $value): bool
    {
        $selections = [
            'string' => is_string($value),
            'numeric' => is_numeric($value),
            'email' => filter_var($value, FILTER_VALIDATE_EMAIL),
            'url' => filter_var($value, FILTER_VALIDATE_URL),
        ];

        if (!isset($selections[$method])) {
            throw new InvalidArgumentException("The method $method is not supported", 422);
        }

        return $selections[$method];
    }

    protected static function check(array $keys, array $payload)
    {
        foreach ($keys as $key => $values) {
            if (!array_key_exists($key, $payload)) {
                throw new InvalidArgumentException("The payload is incomplete. The required key is: $key", 422);
            }

            foreach ($values as $item) {
                if (!self::types($item, $payload[$key])) {
                    throw new InvalidArgumentException("The payload is invalid. The '$key' must be a $item.", 422);
                }
            }
        }
    }

    protected static function validationInquiryIDR(string $privateKey, array $payload)
    {
        $requiredKeys = [
            "merchantCode" => ['string'],
            "orderNum" => ['string'],
            "payMoney" => ['string', 'numeric'],
            "productDetail" => ['string'],
            "notifyUrl" => ['string', 'url'],
            "dateTime" => ['string', 'numeric'],
            "name" => ['string'],
            "phone" => ['string', 'numeric'],
            "expiryPeriod" => ['string', 'numeric'],
        ];

        if (isset($payload['email'])) {
            $requiredKeys['email'] = ['string', 'email'];
        }

        self::check($requiredKeys, $payload);

        if (isset($payload['method']) && !in_array($key = $payload['method'], self::IDR_PAYMENTS)) {
            throw new InvalidArgumentException("The key $key is not supported in the exists array", 422);
        }

        $payload['sign'] = self::sign(self::sortPayload($payload), $privateKey);

        return $payload;
    }

    protected static function validationCredentials(array $payload)
    {
        $requiredKeys = [
            "merchant_id" => ['string'],
            "url_callback" => ['string', 'url'],
            "private_key" => ['string'],
            "public_key" => ['string'],
        ];

        self::check($requiredKeys, $payload);

        return $payload;
    }
}
