<?php

namespace App\Models;

use App\Services\JayaPayment;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class Deposit extends Model
{
    use HasFactory;

    protected $guarded = ['id'];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public static function inquiry(array $payload)
    {
        return self::create($payload);
    }

    public static function paymentSuccess(string $transactionId, array $payload = []): bool
    {
        $data = self::where('transaction_id', $transactionId)->first();
        if (!$data) {
            return false;
        }

        DB::beginTransaction();

        try {
            // Add logging to track the method execution
            Log::info("Starting paymentSuccess for transaction ID: $transactionId");

            $payload['status'] = 'approved';
            $payload['date'] = time();

            $data->update($payload);

            $result = JayaPayment::fee($data->amount, in_array($payload['method_name'], ['MANDIRI', 'PERMATA', 'DANAMON']));

            // Check if a record with the same details already exists to prevent duplication
            $existingLedger = UserLedger::where([
                'user_id' => $data->user_id,
                'reason' => 'payment_approved',
                'perticulation' => sprintf('DEPOSIT #%s', $data->order_id),
                'amount' => $result['net_amount'],
                'debit' => $result['fee'],
                'credit' => $result['amount'],
                'status' => 'approved',
            ])->first();

            if ($existingLedger) {
                Log::warning("Duplicate ledger entry detected for transaction ID: $transactionId");
                DB::rollBack();
                return false;
            }

            UserLedger::create([
                'user_id' => $data->user_id,
                'reason' => 'payment_approved',
                'perticulation' => sprintf('DEPOSIT #%s', $data->order_id),
                'amount' => $result['net_amount'],
                'debit' => $result['fee'],
                'credit' => $result['amount'],
                'status' => 'approved',
            ]);

            $data->user->increment('balance', $result['amount']);

            DB::commit();
            Log::info("PaymentSuccess completed successfully for transaction ID: $transactionId");
            return true;
        } catch (Exception $e) {
            Log::error("Exception occurred in paymentSuccess for transaction ID: $transactionId", ['exception' => $e]);
            DB::rollBack();
            return false;
        }
    }
}
