<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Purchase;
use App\Models\User;
use App\Models\UserLedger;
use Carbon\Carbon;

class CronController extends Controller
{
    public function process(Request $request)
    {
        // 🔒 Cek kunci keamanan
        $key = $request->query('key');
        if ($key !== env('CRON_SECRET_KEY', 'runnn_secure_2025')) {
            return response()->json(['status' => 'error', 'message' => 'Invalid key'], 403);
        }

        $now = Carbon::now();

        // Ambil semua pembelian aktif yang waktunya sudah 24 jam
        $purchases = Purchase::where('status', 'active')
            ->where('date', '<=', $now)
            ->get();

        $count = 0;

        foreach ($purchases as $purchase) {
            $user = User::find($purchase->user_id);
            if (!$user) continue;

            $income = $purchase->daily_income;

            // Tambah saldo user
            $user->increment('balance', $income);

            // Catat di ledger user
            UserLedger::create([
                'user_id'        => $user->id,
                'reason'         => 'daily_income',
                'perticulation'  => 'Pendapatan harian dari produk',
                'amount'         => $income,
                'credit'         => $income,
                'status'         => 'approved',
                'date'           => now(),
            ]);

            // 🔁 Berikan rabat ke 3 level atas
            $this->giveTeamRebate($user, $income);

            // Update tanggal berikutnya (24 jam ke depan)
            $purchase->date = Carbon::now()->addHours(24);

            // Cek masa berlaku paket
            if (Carbon::now()->gte($purchase->validity)) {
                $purchase->status = 'expired';
            }

            $purchase->save();
            $count++;
        }

        return response()->json([
            'status'  => 'success',
            'message' => "✅ Komisi harian & rabat tim berhasil diproses untuk {$count} pengguna.",
        ]);
    }

    /**
     * 🧭 Fungsi Rabat Komisi Tim (Level 1–3)
     */
    private function giveTeamRebate($user, $dailyIncome)
    {
        $rebateLevels = [
            1 => 0.04, // 4%
            2 => 0.02, // 2%
            3 => 0.01, // 1%
        ];

        $current = $user;

        for ($level = 1; $level <= 3; $level++) {
            if (!$current->ref_by) break;

            $upline = User::find($current->ref_by);
            if (!$upline) break;

            $rebateAmount = $dailyIncome * $rebateLevels[$level];

            if ($rebateAmount > 0) {
                $upline->increment('balance', $rebateAmount);

                UserLedger::create([
                    'user_id'                  => $upline->id,
                    'get_balance_from_user_id' => $user->id,
                    'reason'                   => 'team_rebate',
                    'perticulation'            => 'Rabat tim level ' . $level,
                    'amount'                   => $rebateAmount,
                    'credit'                   => $rebateAmount,
                    'status'                   => 'approved',
                    'step'                     => 'level_' . $level,
                    'date'                     => now(),
                ]);
            }

            $current = $upline;
        }
    }
}
