<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Purchase;
use App\Models\User;
use App\Models\UserLedger;
use Carbon\Carbon;

class ProcessDailyIncome extends Command
{
    protected $signature = 'income:daily';
    protected $description = 'Memberikan penghasilan harian otomatis ke saldo user dari produk aktif setelah 24 jam serta rabat tim.';

    public function handle()
    {
        $now = Carbon::now();

        // Ambil semua purchase aktif
        $purchases = Purchase::where('status', 'active')
            ->where('date', '<=', $now)
            ->get();

        foreach ($purchases as $purchase) {
            $user = User::find($purchase->user_id);
            if (!$user) continue;

            // --- Penghasilan harian utama ---
            $income = $purchase->daily_income;
            $user->balance += $income;
            $user->save();

            // Catat ke ledger user
            UserLedger::create([
                'user_id'       => $user->id,
                'reason'        => 'daily_income',
                'perticulation' => 'Hasil investasi harian',
                'amount'        => $income,
                'credit'        => $income,
                'status'        => 'approved',
                'date'          => now(),
            ]);

            // --- Berikan rabat tim ke upline ---
            $this->processTeamRebate($user, $income);

            // Update tanggal berikutnya (setiap 24 jam)
            $purchase->update([
                'date' => $now->copy()->addHours(24),
            ]);

            // Cek masa berlaku produk
            if ($now->greaterThanOrEqualTo($purchase->validity)) {
                $purchase->update(['status' => 'expired']);
            }

            $this->info("Income harian diberikan ke user ID {$user->id} sebesar {$income}");
        }

        $this->info('✅ Proses income harian & rabat tim selesai.');
    }

    /**
     * Berikan rabat tim ke 3 level upline
     * Level 1 → 4%, Level 2 → 2%, Level 3 → 1%
     */
    private function processTeamRebate($user, $income)
    {
        $rebateRates = [0.04, 0.02, 0.01]; // Level 1-3
        $currentUpline = $user->ref_by ?? null;

        foreach ($rebateRates as $level => $rate) {
            if (!$currentUpline) break;

            $upline = User::where('ref_id', $currentUpline)->first();
            if (!$upline) break;

            $rebate = $income * $rate;

            if ($rebate > 0) {
                $upline->balance += $rebate;
                $upline->save();

                UserLedger::create([
                    'user_id'       => $upline->id,
                    'reason'        => 'team_rebate',
                    'perticulation' => 'Rabat Tim Level ' . ($level + 1) . ' dari profit harian downline',
                    'amount'        => $rebate,
                    'credit'        => $rebate,
                    'status'        => 'approved',
                    'date'          => now(),
                ]);

                $this->info("Rabat tim level " . ($level + 1) . " diberikan ke user ID {$upline->id} sebesar {$rebate}");
            }

            // Naik ke upline berikutnya
            $currentUpline = $upline->ref_by ?? null;
        }
    }
}
